
import  ast 
import re 

class VariableNameVisitor(ast.NodeVisitor):
    def __init__(self):
        self.variable_names = []
        self.current_scope = []

    def visit_FunctionDef(self, node):
        # Enter a new scope (function)
        self.current_scope.append(node.name)
        self.generic_visit(node)
        # Exit the current scope (function)
        self.current_scope.pop()
        return node 

    def visit_ClassDef(self, node):
        # Enter a new scope (class)
        self.current_scope.append(node.name)
        self.generic_visit(node)
        # Exit the current scope (class)
        self.current_scope.pop()
        return node 

    def visit_Assign(self, node):
        for target in node.targets:
            if isinstance(target, ast.Name):
                variable_name = target.id
                scope  = ".".join(self.current_scope)
                variable_name = ":".join([scope, variable_name] if scope else [variable_name])
                if variable_name not in self.current_scope:
                    self.variable_names.append(variable_name)
        return node 


class VariableNameWithAttrVisitor(VariableNameVisitor):

    def visit_FunctionDef(self, node):
        # Enter a new scope (function)
        self.current_scope.append("#func#"+node.name)
        self.generic_visit(node)
        # Exit the current scope (function)
        self.current_scope.pop()
        return node 

    def visit_ClassDef(self, node):
        # Enter a new scope (class)
        self.current_scope.append("#class#"+node.name)
        self.generic_visit(node)
        # Exit the current scope (class)
        self.current_scope.pop()
        return node 



def get_varnames_from_code(tree_node ):
    if not isinstance(tree_node,ast.Module):
        return [],[]
    variable_visitor = VariableNameWithAttrVisitor()
    variable_visitor.visit(tree_node)
    var_list = variable_visitor.variable_names
    var_list = list(set(var_list))
    
    var_list_wo_scope = [x.split(":")[-1] for x in var_list ]
    # var_list_wo_scope = list(set(var_list_wo_scope))
    return var_list , var_list_wo_scope 

    
def get_funcnames_from_code(tree_node ,var_list = None  ):
    if var_list is None :
        var_list,_= get_varnames_from_code(tree_node=tree_node)
        # variable_visitor = VariableNameWithAttrVisitor()
        # variable_visitor.visit(tree_node)
        # var_list = variable_visitor.variable_names
    func_list = []
    for input_str in var_list :
        for it in re.finditer("#func#(?:[^.|^:]+)",  input_str ):
            s, e= it.start() , it.end()
            func_list.append( input_str[:e])
    func_list = [x.replace("#class#","").replace("#func#","") for x in func_list ] 
    return func_list 


def get_classnames_from_code(tree_node, var_list=None ):
    if var_list is None :        
        var_list,_= get_varnames_from_code(tree_node=tree_node)
        # variable_visitor = VariableNameWithAttrVisitor()
        # variable_visitor.visit(tree_node)
        # var_list = variable_visitor.variable_names
    func_list = []
    for input_str in var_list :
        for it in re.finditer("#class#(?:[^.|^:]+)",  input_str ):
            s, e= it.start() , it.end()
            func_list.append( input_str[:e])
    func_list = [x.replace("#class#","").replace("#func#","") for x in func_list ] 
    return func_list 



import pickle 
import os 
VAR_NAMES_MOST_COMMON=None
FUNC_NAMES_MOST_COMMON=None


import builtins
import keyword

KEYWORDS = set( dir(builtins) ) .union( set( keyword.kwlist) ).union( set(["self", "super", "Exception", "__init__", "__main__"]) )


def load_cache(path="/tmp/"):
    
    vnames = pickle.load(open(os.path.join(path, "vnames.pkl"), "rb"))
    vnames = [x for (x, _) in vnames.most_common(50000) if x not in KEYWORDS]
    fnames = pickle.load(open(os.path.join(path, "fnames.pkl"), "rb"))
    fnames = [x for (x, _) in fnames.most_common(5000) if x not in KEYWORDS]
    global VAR_NAMES_MOST_COMMON
    global FUNC_NAMES_MOST_COMMON
    
    FUNC_NAMES_MOST_COMMON = fnames
    VAR_NAMES_MOST_COMMON = vnames
    
    
if FUNC_NAMES_MOST_COMMON is None :
    load_cache()
    
