import ast
import astor

import string 
import os 
from typing import List 
import random 
from .utils import get_varnames_from_code as get_all_varnames  
from .utils import get_funcnames_from_code as get_all_funcnames
# from .utils import VAR_NAMES_MOST_COMMON 
from .utils import VAR_NAMES_MOST_COMMON,FUNC_NAMES_MOST_COMMON

import libcst

class RenameTransformer(libcst.CSTTransformer):
    def __init__(self, rename_pairs):
        self.rename_pairs = rename_pairs
        self.restore_keywords = []

    def _rename(self, original_node, renamed_node):
        if original_node.value in self.rename_pairs.keys():
            return renamed_node.with_changes(value=self.rename_pairs[original_node.value])
        else:
            return renamed_node

    def leave_Name(self, original_node, renamed_node):
        """
        LibCST parses variables in f-strings.
        All [Name] nodes are already parsed correctly.
        """
        return self._rename(original_node, renamed_node)

    def visit_Arg(self, node):
        """
        LibCST does not provide APIs to decide whether to visit the child attributes or not.
        Some tricky hack is needed, e.g. restoring the original keyword as follow.
        """
        if node.keyword and node.keyword.value in self.rename_pairs.keys():
            self.restore_keywords.append(node.keyword.value)
        return True

    def leave_Arg(self, original_node, renamed_node):
        try:
            restore = self.restore_keywords.pop()
            return renamed_node.with_changes(keyword=renamed_node.keyword.with_changes(value=restore))
        except IndexError:
            # stack is empty
            return renamed_node

def generate_random_name(count=1, name_set=set()):
    if type(name_set)!=set :
        name_set = set(name_set)
        
    rand_names = random.sample(VAR_NAMES_MOST_COMMON,count )
    if len( set(rand_names).intersection(name_set)  )>0:
        return generate_random_name(count=count, name_set=name_set )
    return rand_names




def get_renames_paris(var_list:List[str],rate=0.2 ):
    def build_replace_name(input_str,replacement_char):
        # replacement_char = generate_random_name()
        if ":" in input_str : 
            import re
            result_str = re.sub(r':.+', ':' + replacement_char, input_str)
            return result_str 
        else:
            return replacement_char
        
    candicate_keys = random.sample(var_list, round(rate*len(var_list) ) )
    replacement_keys =generate_random_name(count=len(candicate_keys)) #  random.sample(var_list, int(rate*len(var_list) ) )
    
    rename_pairs = {input_str:build_replace_name(input_str,replacement_char)  for input_str,replacement_char in zip(candicate_keys,replacement_keys) }
    return rename_pairs 



def mt_rename(python_code  ,rename_candidates=None  , rate=0.2  ):
    
    if rename_candidates is None :
        tree_node = ast.parse(python_code)
        _ , all_vars = get_all_varnames(tree_node=tree_node)
        all_vars = sorted(all_vars)
        rename_candidates = get_renames_paris(var_list=all_vars , rate=rate )

    rename_transformer = RenameTransformer(rename_candidates)
    original_tree = libcst.parse_module(python_code)
    modified_tree_node = original_tree.visit(rename_transformer)

    return modified_tree_node.code


def mt_func_rename(python_code  ,rename_candidates=None  , rate=0.2 ):
    
    if rename_candidates is None :
        tree_node = ast.parse(python_code)
        all_vars = get_all_funcnames(tree_node=tree_node)
        all_vars = sorted(all_vars)
        rename_candidates = get_renames_paris(var_list=all_vars , rate=rate )

    rename_transformer = RenameTransformer(rename_candidates)
    original_tree = libcst.parse_module(python_code)
    modified_tree_node = original_tree.visit(rename_transformer)

    return modified_tree_node.code



