import asyncio
import functools
import logging
import ssl
import time
from enum import Enum
from typing import Dict, List
from typing import Optional,Union

import msgpack
from pydantic import BaseModel, validator

import pika
from pika.exceptions import AMQPConnectionError

logging.basicConfig(format='%(levelname)s:%(message)s', level=logging.ERROR)
logger = logging.getLogger(__name__)


def sync(f):
    @functools.wraps(f)
    def wrapper(*args, **kwargs):
        return asyncio.get_event_loop().run_until_complete(f(*args, **kwargs))

    return wrapper


class Priority(Enum):
    LOW = 1
    NORMAL = 5
    HIGH = 10


class Headers(BaseModel):
    job_id: Optional[str] = None
    priority: Optional[int] = None
    task_type: Optional[str] = None

    # @validator("priority", pre=True)
    # def _convert_priority(self, value):
    #     return Priority[value]

class MsgBody(BaseModel):
    id: str
    content: Optional[Union[str,Dict,List] ]
    mapping_key: Optional[Union[str,Dict,List] ] = None
    task: Optional[Union[str,Dict,List] ] = None
    # tag: Optional[str] = None
    pipeline: Optional[Union[str,Dict,List] ] = None

    result: Optional[Union[str,Dict,List] ] = None
    data: Optional[Union[str,Dict,List] ] = None


class RabbitMQConfig(BaseModel):
    host: str
    port: int
    username: str
    password: str
    protocol: str


import random 
    
class BasicPikaClient:
    def __init__(self):
        self.parameters_list=[]
        self._init_connection_parameters()
        self._connect()

    def _connect(self):
        tries = 0
        while True:
            try:
                rid = random.randint(0,len(self.parameters_list) )
                self.parameters = self.parameters_list[rid ]
                
                self.connection = pika.BlockingConnection(self.parameters)
                self.channel = self.connection.channel()
                if self.connection.is_open:
                    break
            except (AMQPConnectionError, Exception) as e:
                time.sleep(5)
                tries += 1
                if tries == 20:
                    raise AMQPConnectionError(e)

    def _init_connection_parameters(self):
        
        # host_list=["CSLS60U.dynip.ntu.edu.sg","10.96.184.147","csl-server2.dynip.ntu.edu.sg","csl-server3.dynip.ntu.edu.sg"]
        # host_list=["CSLS60U.dynip.ntu.edu.sg","csl-server2.dynip.ntu.edu.sg","csl-server3.dynip.ntu.edu.sg"]
        host_list=["csl-server11.dynip.ntu.edu.sg", "csl-svr201.dynip.ntu.edu.sg" , ] 
        # host_list=["155.69.145.160","155.69.151.252"]
        for host_list_one in host_list:
            self.username = "detection"
            self.password = "detection123"
            # self.host = "155.69.146.118"
            # self.host = host_list[ random.randint(0,len(host_list)-1 ) ]
            self.host = host_list_one 
            self.port = 5672
            self.protocol = "amqp"
            self.credentials = pika.PlainCredentials(self.username, self.password)
            parameters = pika.ConnectionParameters(
                self.host,
                int(self.port),
                "/detection",
                self.credentials,
            )
            if self.protocol == "amqps":
                # SSL Context for TLS configuration of Amazon MQ for RabbitMQ
                ssl_context = ssl.SSLContext(ssl.PROTOCOL_TLSv1_2)
                ssl_context.set_ciphers("ECDHE+AESGCM:!ECDSA")
                parameters.ssl_options = pika.SSLOptions(context=ssl_context)

            self.parameters_list.append(parameters)
        
    def check_connection(self):
        if not self.connection or self.connection.is_closed:
            self._connect()

    def close(self):
        self.channel.close()
        self.connection.close()

    def declare_queue(
        self, queue_name, exclusive: bool = False, max_priority: int = 10
    ):
        self.check_connection()
        logger.debug(f"Trying to declare queue({queue_name})...")
        self.channel.queue_declare(
            queue=queue_name,
            exclusive=exclusive,
            durable=True,
            arguments={"x-max-priority": max_priority},
        )

    def declare_exchange(self, exchange_name: str, exchange_type: str = "direct"):
        self.check_connection()
        self.channel.exchange_declare(
            exchange=exchange_name, exchange_type=exchange_type
        )

    def bind_queue(self, exchange_name: str, queue_name: str, routing_key: str):
        self.check_connection()
        self.channel.queue_bind(
            exchange=exchange_name, queue=queue_name, routing_key=routing_key
        )

    def unbind_queue(self, exchange_name: str, queue_name: str, routing_key: str):
        self.channel.queue_unbind(
            queue=queue_name, exchange=exchange_name, routing_key=routing_key
        )


class BasicMessageSender(BasicPikaClient):
    def encode_message(self, body: Dict, encoding_type: str = "bytes"):
        if encoding_type == "bytes":
            return msgpack.packb(body)
        else:
            raise NotImplementedError

    def send_message(
        self,
        exchange_name: str,
        routing_key: str,
        body: Dict,
        headers: Optional[Headers],
    ):
        self.check_connection()
        body = self.encode_message(body=body)
        self.channel.basic_publish(
            exchange=exchange_name,
            routing_key=routing_key,
            body=body,
            properties=pika.BasicProperties(
                delivery_mode=pika.spec.PERSISTENT_DELIVERY_MODE,
                priority=headers.priority,
                headers=headers.dict(),
            ),
        )
        logger.debug(
            f"Sent message. Exchange: {exchange_name}, Routing Key: {routing_key}, Body: {body[:128]}"
        )


class BasicMessageReceiver(BasicPikaClient):
    def __init__(self):
        super().__init__()
        self.channel_tag = None

    def decode_message(self, body):
        if type(body) == bytes:
            return msgpack.unpackb(body)
        else:
            raise NotImplementedError

    def get_message(self, queue_name: str, auto_ack: bool = False):
        method_frame, header_frame, body = self.channel.basic_get(
            queue=queue_name, auto_ack=auto_ack
        )
        if method_frame:
            logger.debug(f"{method_frame}, {header_frame}, {body}")
            return method_frame, header_frame, body
        else:
            logger.debug("No message returned")
            return None
    def ack_done(self,tag ):
        self.check_connection()

        self.channel.basic_ack(delivery_tag=tag )
    def ack_fale(self,tag ):
        self.check_connection()

        self.channel.basic_nack(delivery_tag=tag )
        
    def consume_messages(self, queue, callback):
        self.channel.basic_qos(prefetch_count=1)

        import uuid 
        consumer_tag = str(uuid.uuid4())
        
        self.check_connection()
        self.channel_tag = self.channel.basic_consume(
            queue=queue, on_message_callback=callback, auto_ack=False,consumer_tag=consumer_tag,
        )
        logger.debug(" [*] Waiting for messages. To exit press CTRL+C")
        self.channel.start_consuming()

    def cancel_consumer(self):
        if self.channel_tag is not None:
            self.channel.basic_cancel(self.channel_tag)
            self.channel_tag = None
        else:
            logger.error("Do not cancel a non-existing job")
